/* resolver.c - Copyright (C) 2004 Pat Thoyts <patthoyts@users.sourceforge.net>
 *
 * This is a resolver process. This program reads hostnames from standard
 * input and uses the standard system resolver to obtain the IP address (or
 * addresses) and returns these as a line on standard output.
 *
 * $Id: resolver.c,v 1.2 2004/08/26 09:49:28 pat Exp $
 */

#if defined(_UNICODE) && !defined(UNICODE)
#define UNICODE
#endif

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <winsock2.h>
#include <ws2tcpip.h>
#include <tchar.h>
#include <stdio.h>

#if _MSC_VER >= 1000
#pragma comment(lib, "ws2_32")
#endif

void Win32Error(FILE *fp, LPCTSTR sz, HRESULT hr);
void DumpData(LPCTSTR szLabel, LPBYTE pData, DWORD cbData);

int
_tmain(int argc, LPCTSTR argv[])
{
    WSADATA wsd;
    DWORD dwCookie = 0;

    if (WSAStartup(MAKEWORD(2,2), &wsd) != 0) {
        Win32Error(stderr, _T("WSAStartup failed"), GetLastError());
        return -1;
    }

    if (LOBYTE(wsd.wVersion) != 2 ||
        HIBYTE(wsd.wVersion) != 2) {
        _ftprintf(stderr, 
                  _T("error: failed to load a compatible winsock version\n"));
        WSACleanup();
        return -1;
    }
    
    while (1) {
        struct addrinfo hints = {0};
        struct addrinfo *res = NULL;
        LPSTR hostname;
        TCHAR buffer[1024];
        LPTSTR line, q, tname;
        int r = 0, nc = 0;
    
        line = _fgetts(buffer, 1024, stdin);
        if (line == NULL)
            break;
        
        /* trim whitespace */
        while(*line && _istspace(*line))
            line++;
        q = line + lstrlen(line) - 1;
        while (q > line && *q && isspace(*q))
            q--;
        *++q = 0;

#ifdef _UNICODE
        nc = WideCharToMultiByte(CP_ACP, 0, line, -1, NULL, 0, NULL, NULL);
        hostname = (LPSTR)_alloca(nc + 1);
        WideCharToMultiByte(CP_ACP, 0, line, -1, hostname, nc, NULL, NULL);
#else
        hostname = line;
#endif

        dwCookie++;
        ZeroMemory(&hints, sizeof(hints));
        hints.ai_family = PF_UNSPEC;
        hints.ai_flags = AI_CANONNAME;
        hints.ai_socktype = SOCK_STREAM;
        r = getaddrinfo(hostname, "", &hints, &res);

        if (r != 0) {
            Win32Error(stdout, _T("error"), WSAGetLastError());
        } else {
            struct addrinfo *p = res;
            while (p != NULL) {
                char name[NI_MAXHOST];
                /*
                printf("fam:%x prot:%x len:%d res:%s\n", 
                       p->ai_family, p->ai_protocol, p->ai_addrlen,
                       p->ai_canonname);
                DumpData(_T("addrinfo"), (LPBYTE)p, sizeof(struct addrinfo));
                */
                getnameinfo(p->ai_addr, p->ai_addrlen, 
                            name, NI_MAXHOST,
                            NULL, 0,
                            NI_NUMERICHOST | NI_NUMERICSERV);
#ifdef _UNICODE
                nc = MultiByteToWideChar(CP_ACP, 0, name, -1, NULL, 0);
                tname = (LPWSTR)_alloca(nc + sizeof(WCHAR));
                MultiByteToWideChar(CP_ACP, 0, name, -1, tname, nc);
#else
                tname = name;
#endif
                _tprintf(_T("%s "), tname);
                p = p->ai_next;
            }
            _tprintf(_T("\n"));
            freeaddrinfo(res);
        }

        fflush(stdout);
    }

    WSACleanup();
    return 0;
}

void
Win32Error(FILE *fp, LPCTSTR szMessage, HRESULT hr)
{
    LPTSTR lpBuffer = NULL;
    DWORD  dwLen = 0;
    
    dwLen = FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER 
                          | FORMAT_MESSAGE_FROM_SYSTEM,
                          NULL, (DWORD)hr, LANG_NEUTRAL,
                          (LPTSTR)&lpBuffer, 0, NULL);
    if (dwLen < 1) {
        dwLen = FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER 
                              | FORMAT_MESSAGE_FROM_STRING
                              | FORMAT_MESSAGE_ARGUMENT_ARRAY,
                              _T("code 0x%1!08X!%n"), 0, LANG_NEUTRAL,
                              (LPTSTR)&lpBuffer, 0, (va_list *)&hr);
    }
    _ftprintf(fp, _T("%s"), szMessage);
    if (dwLen > 0) {
        _ftprintf(fp, _T(": "));
        _ftprintf(fp, lpBuffer);
    }
    LocalFree((HLOCAL)lpBuffer);
}

void
DumpData(LPCTSTR szLabel, LPBYTE pData, DWORD cbData)
{
    TCHAR line[120];
    DWORD cn = 0;
    LPTSTR p = line, q = line + 40;
    FillMemory(line, 80 * sizeof(TCHAR), 32);
    _sntprintf(p, 80, _T("%s:"), szLabel);
    p += lstrlen(szLabel) + 1;
    *p = 32;
    for (cn = 0; cn < cbData; cn++) {
        if (cn % 16 == 0) {
            _sntprintf(q, 80, _T("\n% 4d: "), cn);
            p = line, q = line + 40;
            _ftprintf(stdout, line);
            FillMemory(line, 80 * sizeof(TCHAR), 32);
        }
        _stprintf(p, _T("%02x"), (int)pData[cn]);
        _stprintf(q, _T("%c"), 
                   _istprint(pData[cn]) ? (int)pData[cn] : _T('.'));
        p += 2;
        q += 1;
        *p = _T(' ');
    }
    _stprintf(q, _T("\n"));
    _ftprintf(stdout, line);
}
